/*
             LUFA Library
     Copyright (C) Dean Camera, 2010.
              
  dean [at] fourwalledcubicle [dot] com
      www.fourwalledcubicle.com
*/

/*
  Copyright 2010  Dean Camera (dean [at] fourwalledcubicle [dot] com)

  Permission to use, copy, modify, distribute, and sell this 
  software and its documentation for any purpose is hereby granted
  without fee, provided that the above copyright notice appear in 
  all copies and that both that the copyright notice and this
  permission notice and warranty disclaimer appear in supporting 
  documentation, and that the name of the author not be used in 
  advertising or publicity pertaining to distribution of the 
  software without specific, written prior permission.

  The author disclaim all warranties with regard to this
  software, including all implied warranties of merchantability
  and fitness.  In no event shall the author be liable for any
  special, indirect or consequential damages or any damages
  whatsoever resulting from loss of use, data or profits, whether
  in an action of contract, negligence or other tortious action,
  arising out of or in connection with the use or performance of
  this software.
*/

/*
  All changes to this file are Copyright (C) BitWizard.
  info@bitwizard.nl
  
  www.bitwizard.nl
  www.bitwizard.nl/shop
  www.bitwizard.nl/wiki
*/

#include <util/delay.h>

/** \file
 *
 *  Main source file for the VirtualSerial demo. This file contains
 *  the main tasks of the demo and is responsible for the initial
 *  application hardware configuration.
 */
 
#include "VirtualSerial.h"
#include "bits.h"
#include "date.h"
#include <avr/io.h>
/** LUFA CDC Class driver interface configuration and state
 *  information. This structure is passed to all CDC Class driver
 *  functions, so that multiple instances of the same class within a
 *  device can be differentiated from one another.
 */
USB_ClassInfo_CDC_Device_t VirtualSerial_CDC_Interface =
  {
    .Config =
    {
      .ControlInterfaceNumber         = 0,

      .DataINEndpointNumber           = CDC_TX_EPNUM,
      .DataINEndpointSize             = CDC_TXRX_EPSIZE,
      .DataINEndpointDoubleBank       = false,

      .DataOUTEndpointNumber          = CDC_RX_EPNUM,
      .DataOUTEndpointSize            = CDC_TXRX_EPSIZE,
      .DataOUTEndpointDoubleBank      = false,

      .NotificationEndpointNumber     = CDC_NOTIFICATION_EPNUM,
      .NotificationEndpointSize       = CDC_NOTIFICATION_EPSIZE,
      .NotificationEndpointDoubleBank = false,
    },
  };


static unsigned char tick, tock;


/** Standard file stream for the CDC interface when set up, so that
 *  the virtual CDC COM port can be used like any regular character
 *  stream in the C APIs
 */
static FILE USBSerialStream;

void delay_ms(unsigned int ms)
/* delay for a minimum of <ms> */
{
  // we use a calibrated macro. This is more
  // accurate and not so much compiler dependent
  // as self made code.
  while(ms){
    _delay_ms(0.98);
    ms--;
  }
}


void hexnibble (unsigned char b)
{
  b += '0';
  if (b <= '9') {
    fputc (b, &USBSerialStream);
    return;
  }
  b += 7;
  fputc (b, &USBSerialStream);
  return;
}

void hexbyte (unsigned char v)
{
  hexnibble (v >> 4);
  hexnibble (v & 0xf);
}

void hexshort (unsigned int v)
{
  hexbyte (v >> 8);
  hexbyte (v & 0xff);
}


unsigned char lbp;
char inputbuffer[0x20];

unsigned char lb_peekchar (void)
{
  return inputbuffer[lbp];
}

unsigned char lb_getchar (void)
{
  return inputbuffer[lbp++];
}

unsigned char conv_nibble (unsigned char c)
{
  if (c >= 'A') {
    return (c & ~0x20) - 'A' + 10;
  } 
  return c - '0';
}


int lb_gethex(void)
{
  int v = 0;
  unsigned char c;

  while (lb_peekchar() == ' ') lb_getchar (); 
  
  while ( ((c = lb_getchar ()) != ' ') &&
          (c != '\n') && 
          (c != '\r')) {
    v = (v << 4) | conv_nibble (c);
  }
  return v; 
}

#define LED1 pC2
#define LED2 pD0
#define LP_R pB3
#define LP_G pB2
#define LP_B pB5
#define OE   pD5
#define SER0 pD1
#define SER1 pD2
#define SER2 pD3
#define SER3 pD4
#define LATCH pD6
#define RESET pB1
#define CLOCK pB0 // shiftclock
#define CLK_IN pB4   // 50 hz in
#define IO0 pB7
#define IO1 pB6
#define IO2 pC6
#define IO3 pC7

unsigned char rdata[8];
unsigned char gdata[8];
unsigned char bdata[8];
unsigned char quadrant[4] = {SER0, SER3, SER2, SER1};
short milliseconds;
unsigned char hours;
unsigned char minutes;
unsigned char seconds;
unsigned char mode = 2;


short brightness[3] = {0x7800, 0x7e00, 0x0};
short intlength;


void process_line (char *inputbuffer)
{
  unsigned short i;
  int c;

  switch (lb_getchar()) {
  /** Debugging commands, these only work correctly if you prevent the loopfunc
   * from calling the display_time routine, and modify the 
   * "ISR (TIMER1_COMPB_vect)" routine to not automatically scan the 
   * colors.
   *
   * TODO: Make a command to switch (debugging)modes: -- Mode1 doesn't work yet.
   *   Mode1: PWM routine disabled, bitmap generation disabled
   *   Mode2: PWM routine enabled, bitmap generation disabled
   *   Mode3: PWM routine enabled, bitmap generation enabled (normal operation)
   *   
   */
  case 'b':                     //Only turn on blue LEDs
    clr_bit (LP_R);
    clr_bit (LP_G);
    set_bit (LP_B);
    break;
  case 'd':                     //Write identical data to all 4 shift registers
    c = lb_gethex ();
    for (i=0; i<15; i++) {
      if (i==8) {               //LED8 is not used, so generate an extra clock-
        set_bit (CLOCK);        //pulse
        clr_bit (CLOCK);
      }
      if (c&1) {
        set_bit (SER0);
        set_bit (SER1);
        set_bit (SER2);
        set_bit (SER3);
      } else {
        clr_bit (SER0);
        clr_bit (SER1);
        clr_bit (SER2);
        clr_bit (SER3);
      }
      set_bit (CLOCK);
      c=c>>1;
      clr_bit (CLOCK);
      set_bit (LATCH);    
      clr_bit (LATCH);
    }
    break;
  case 'D':                     //Manually set rdata (example: 
    rdata[0]=lb_gethex ();      //"D ffff ffff ffff" turn on all red LEDs)
    rdata[1]=lb_gethex ();
    rdata[2]=lb_gethex ();
    rdata[3]=lb_gethex ();
    break;
  case 'E':                     //Manually set gdata
    gdata[0]=lb_gethex ();
    gdata[1]=lb_gethex ();
    gdata[2]=lb_gethex ();
    gdata[3]=lb_gethex ();
    break;
  case 'f': //reset             //Flush input registers
    clr_bit (RESET);
    set_bit (RESET);
    break;
  case 'F':                     //Manually set bdata
    bdata[0]=lb_gethex ();
    bdata[1]=lb_gethex ();
    bdata[2]=lb_gethex ();
    bdata[3]=lb_gethex ();
    break;
  case 'g':                     //Only turn on green LEDs
    clr_bit (LP_R);
    clr_bit (LP_B);
    set_bit (LP_G);
    break;
  case 'l':                     //Latch data to output registers
    set_bit (LATCH);    
    clr_bit (LATCH);
    break;
  case 'o':                     //SET OE bit (Outputs in tri-state)
    set_bit (OE);
    break;
  case 'O':
    clr_bit (OE);               //Clear OE bit (outputs driven)
    break;
  case 'r':                     //Only turn on red LEDs
    clr_bit (LP_G);
    clr_bit (LP_B);
    set_bit (LP_R);
    break;
  /** End of debugging commands */
  
/*  case 'm':
    c = lb_gethex ();
    if (c > 2) {
      fputs (" Error: Mode ", &USBSerialStream);
      hexbyte (c);
      fputs ("does not exist.", &USBSerialStream);
      break;
    }
    mode = c;
    break;*/
  
  case 'B':                     
    // Adjust the white balance, in parts of a 48000(dec) tick cycle 
    // (default: 0x7800, 0x7e00, 0x0, so type: "B 7800 7e00 0")
    for (i=0; i<3; i++) {
      brightness[i] = lb_gethex ();
    }
    OCR1B = brightness[0];
    break;
  case 'i':                     //Only print the milliseconds of the current 
        hexshort (milliseconds);//time
    break;
  case 'I':
    fputs (" ", &USBSerialStream);
    hexshort (intlength);
    break;
  case 't':                     //Print the current time in hex.
    fputs (" ", &USBSerialStream);
    hexbyte (hours);
    fputs (":", &USBSerialStream);
    hexbyte (minutes);
    fputs (":", &USBSerialStream);
    hexbyte (seconds);
    fputs (".", &USBSerialStream);
    hexshort (milliseconds);
    break;
  case 'T':                     //Set the time, in hex.
    milliseconds=0;             //For example: 14:10:15 = "T E A F"
    hours=lb_gethex ();
    minutes=lb_gethex ();
    seconds=lb_gethex ();
    break;
  case 'z':                     //Jump to bootloader
    //UsbDisable ();
    __asm ("jmp 0x3000;");
    break;
  }
}


/** Light LED [num] from color [color]
 *  
 *  
 */
void lightled (unsigned char *color, unsigned char num)
{
  unsigned char i;

  for (i=0; i<8; i++) {               //Clear the bitmap
    color[i]=0;
  }
  num = (num+29)%60;
  num = num + (num+7)/15;
  color[(num/2)%8]=(0x08<<4*(0x1&num)) >> (num>>4);
}


/** This routine prepares the right bitmap for each color.
 *
 *
 */
void display_time (unsigned char hours, unsigned char minutes, 
                   unsigned char seconds)
{
//   if (mode == 2) {
  if (1) {
    lightled (rdata, seconds);
    lightled (gdata, minutes);
    lightled (bdata, ((hours%12 *5) + (minutes/12)));
  }
}


/** Checks if a button is presed, and does what needs to be done
 *
 *  The menu structure:
 *  -Set time (button for hour+1, minutes+1, seconds=0)
 * TODO:   -Set white-balance
 *  
 *  
 */
void checkbuttons (void)
{
  static short ptime;
  static short follow_ms;
  
  while (follow_ms != milliseconds) {
    follow_ms++;
    if (follow_ms > 999) follow_ms -= 1000;
    ptime--;
  }
  
  if (get_input (IO2) == 0) {               //IO1
    seconds = 0;
    display_time (hours, minutes, seconds);
  } else if (get_input(IO3) == 0) {         //IO2
    if (ptime <= 0) {
      ptime = 250;
      minutes++;
      display_time (hours, minutes, seconds);
    }
  } else if (get_input(IO0) == 0) {         //IO2
    if (ptime <= 0) {
      ptime = 700;
      hours++;
      display_time (hours, minutes, seconds);
    }
  }
}


void do_loopfunc (void)
{
  static unsigned char oldseconds;
  static int t;

  checkbuttons ();


  while (tock != tick) {
    tock++;
  
    /** Timekeeping routine
     *
     *
     */
    
    milliseconds = milliseconds + 3;
    if (milliseconds>=1000) {
      milliseconds=milliseconds-1000;
      seconds++;
    }
    
    if (seconds>=60) {
      seconds=seconds-60;
      minutes++;
    }
    
    if (minutes>=60) {
      minutes=minutes-60;
      hours++;
    }
    
    if (hours>=24) {
      hours=hours-24;
      //days++;
    }
    
    if (seconds!=oldseconds) {
      oldseconds = seconds;
      display_time(hours, minutes, seconds);
      
    }
  }


  t++;
  if (t > 5000) set_bit (LED1);
  else          clr_bit (LED1);
  if (t > 10000) t = 0;
}



ISR (TIMER1_COMPA_vect)
{
  tick++;
}

  
ISR (TIMER1_COMPB_vect)
{
  unsigned char i;
  static unsigned char color;
  unsigned char opd, *p;
  
/** LED driving routine
 *
 *
 */

//   if (mode != 1) {
  if (1) {
    switch (color) {                             //Select the correct bitmap
      case 0:
        p = rdata;
        break;
      case 1:
        p = gdata;
        break;
      case 2:
        p = bdata;
        break;
      default:
        p = NULL;
        break;
    }
  }

//   if (mode != 1) {
  if (1) {
    opd = PORTD & 0xe1;
    for (i=0; i<8; i++) {
      PORTD = opd | ((*p & 0xf) << 1);
      set_bit (CLOCK);
      clr_bit (CLOCK);
      PORTD = opd | ((*p & 0xf0) >> 3);
      set_bit (CLOCK);
      clr_bit (CLOCK);
      p++;
    }

    clr_bit (LP_R);       //Turn off all FETs
    clr_bit (LP_G);       //
    clr_bit (LP_B);       //
    set_bit (LATCH);      //Latch the shifted data to the outputregisters
    clr_bit (LATCH);      //
  }
  
//   if (mode != 1) {
  if (1) {
    switch (color) {      //Enable the right color.
      case 0:
        set_bit(LP_R);
        break;
      case 1:
        set_bit(LP_G);
        break;
      case 2:
        set_bit(LP_B);
        break;
    }
  }
  
  intlength =  TCNT1 - OCR1B;
  
  
//   if (mode != 1) {
  if (1) {
    OCR1B = brightness[color];
  }
  
//   if (mode != 1) {
  if (1) {
    if (color == 2) {
      color=0;              //Select the next color
    } else {
      color++;              //
    }
  }

}


/** Main program entry point. This routine contains the overall
 *  program flow, including initial setup of all components and the
 *  main program loop.
 */
int main(void)
{
  int c;

  int nchars;
  unsigned char intro_sent;

  SetupHardware();
        
  /* Create a regular character stream for the interface 
  so that it can be used with the stdio.h functions */
  CDC_Device_CreateStream(&VirtualSerial_CDC_Interface, &USBSerialStream);

  nchars = 0;

  OCR1A = 48000-1; // reload counter/interrupt every 3ms. 

  TCCR1A = 0;
  TCCR1B = (1 << CS10) | (1 << WGM12); // Mode 4 CTC. 
  TCCR1C = 0;

  TIMSK1 = _BV (OCIE1A) | _BV (OCIE1B);
  
// #define LED1 pC2
// #define LED2 pD0
// #define LP_R pB3
// #define LP_G pB2
// #define LP_B pB5
// #define OE   pD5
// #define SER0 pD1
// #define SER1 pD2
// #define SER2 pD3
// #define SER3 pD4
// #define LATCH pD6
// #define RESET pB1
// #define CLOCK pB0 // shiftclock
// #define CLK_IN pB4   // 50 hz in
// #define IO0 pB7
// #define IO1 pB6
// #define IO2 pC6
// #define IO3 pC7

  make_output (LED1);
  make_output (LED2);
  make_output (LP_R);
  make_output (LP_G);
  make_output (LP_B);
  make_output (OE);
  make_output (SER0);
  make_output (SER1);
  make_output (SER2);
  make_output (SER3);
  make_output (LATCH);
  make_output (RESET);
  make_output (CLOCK);
  
  make_input (CLK_IN);
  make_input (IO0);
  make_input (IO1);
  make_input (IO2);
  make_input (IO3);
  
  intro_sent = 0;
  
  clr_bit (RESET);      //Flush the input buffers of the shift registers
  set_bit (RESET);
  set_bit (LATCH);      //Latch the empty input buffers of the shift
  clr_bit (LATCH);      //registers to the output registers
  clr_bit (OE);         //Enable the output drivers of the shift registers
  
  sei (); 

  for (;;) {
    if (VirtualSerial_CDC_Interface.State.ControlLineStates.HostToDevice & 
        CDC_CONTROL_LINE_OUT_DTR) {
      if (!intro_sent) {
        fputs("CDC " MYNAME " " DATE"\r\n> ", &USBSerialStream);
      }
      intro_sent = 1;
    }else {
      intro_sent = 0;
    }

    c = CDC_Device_ReceiveByte(&VirtualSerial_CDC_Interface);

    if (c > 0) {
      if (c == 0x7f) {// del
        if (nchars > 0) {
          nchars --;
          fputs ("\b \b", &USBSerialStream);
        }
      } else if (nchars > 0x1e){
        CDC_Device_SendByte (&VirtualSerial_CDC_Interface, '\b');
        nchars = 0; 
      } else {
        inputbuffer[nchars++] = c;
        if ((c == '\r') || 
            (c == '\n')) {
          lbp = 0;
          process_line (inputbuffer);
          fputs ("\r\n> ", &USBSerialStream);
          nchars = 0;
        } else {
          CDC_Device_SendByte (&VirtualSerial_CDC_Interface, c);
        }
      }
    }
    //test3
    CDC_Device_USBTask(&VirtualSerial_CDC_Interface);
    USB_USBTask();
    _delay_ms (0.1);
    do_loopfunc ();
  }
}


/** Configures the board hardware and chip peripherals for the demo's
 *  functionality. */
void SetupHardware(void)
{
  /* Disable watchdog if enabled by bootloader/fuses */
  MCUSR &= ~(1 << WDRF);
  wdt_disable();

  /* Disable clock division */
  clock_prescale_set(clock_div_1);

  /* Hardware Initialization */
  USB_Init();

}

/** Event handler for the library USB Connection event. */
void EVENT_USB_Device_Connect(void)
{
}

/** Event handler for the library USB Disconnection event. */
void EVENT_USB_Device_Disconnect(void)
{
}

/** Event handler for the library USB Configuration Changed event. */
void EVENT_USB_Device_ConfigurationChanged(void)
{

        if (!(CDC_Device_ConfigureEndpoints(&VirtualSerial_CDC_Interface))) {
//        LEDs_SetAllLEDs(LEDMASK_USB_ERROR);
        /* nothing now */
        
        }
}

/** Event handler for the library USB Unhandled Control Request event. */
void EVENT_USB_Device_UnhandledControlRequest(void)
{
        CDC_Device_ProcessControlRequest(&VirtualSerial_CDC_Interface);
}


/** Event handler for the library USB Unhandled Control Request event. */
void EVENT_USB_Device_ControlRequest(void)
{
       CDC_Device_ProcessControlRequest(&VirtualSerial_CDC_Interface);
}

